#ifndef __CClass__
#define __CClass__

/**
 * @file CClass.hpp
 * @brief Definition of a class object.
 */

//	===========================================================================

#include "StdDefines.h"
#include "StdIncludes.h"
#include "CClassManager.hpp"

using Exponent::Basics::CClassManager;

//	===========================================================================

namespace Exponent
{
	namespace Basics
	{
		/**
		 * @class CClass CClass.hpp
		 * @brief Class representation class
		 *
		 * CClass stores information about a class. Every object in the exponent\n
		 * API that is dynamically available (ie - non static classes, and classes\n
		 * without entirely static functions should have a CClass\n
		 * Currently CClass only stores the name of the class, this will change in future\n
		 * versions of the API.\n
		 * Macros are setup so that you can easily implement the correct style of class handling\n
		 * When you write a class, simply perform the following steps:\n
		 * <OL>
		 * 	<LI> Include the macro EXPONENT_CLASS_DECLARATION <B>BEFORE</B> the openeing public statement. Ie it should be the first thing in your classes opening bracket\n
		 * 	<LI> In the class cpp file place the macro EXPONENT_CLASS_IMPLEMENTATION at the top of the file\n
		 * 	<LI> In the case of a template class use the macro EXPONENT_TEMPLATE_CLASS_IMPLEMENTATION instead\n
		 * 	<LI> Inside the constructor place the macro EXPONENT_CLASS_CONSTRUCTION (This ensure that the correct class count is maintained)\n
		 * 	<LI> Inside the destructor place the macro  EXPONENT_CLASS_DESTRUCTION (This ensure that the correct class count is maintained)\n
		 * </OL>
		 * \n
		 * Code is better than a thousand bullet points ;)\n
		 * \n
 		 * Header (myClass.hpp)
		 * @code
		 * class myClass : public CCountedObject
		 * {
		 *      EXPONENT_CLASS_DECLARATION;
		 * public:
		 *
		 *      // Whatever
		 *      ...
		 * };
		 * @endcode
		 *\n
		 * Implementation (myClass.cpp)
		 * @code
		 * EXPONENT_CLASS_IMPLEMENTATION(myClass, myClassParent);
		 *
		 * myClass::myClass()
		 * {
		 *     EXPONENT_CLASS_CONSTRUCTION(myClass);
		 *      // Whatever
		 *      ...
		 * }
		 *
		 * myClass::~myClass()
		 * {
		 *     EXPONENT_CLASS_DESTRUCTION(myClass);
		 *	   // Whatever
		 *     // ...
		 * }
		 * @endcode
		 *
		 * @see CClassManager
		 * @see EXPONENT_CLASS_DECLARATION
		 * @see EXPONENT_CLASS_IMPLEMENTATION
		 * @see EXPONENT_CLASS_CONSTRUCTION
		 * @see EXPONENT_CLASS_DESTRUCTION
		 * @see EXPONENT_TEMPLATE_CLASS_IMPLEMENTATION
		 *
		 * @date 28/02/2005
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 * @version 1.1.0 Converted to split class/header files to ensure proper handling in other classes
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CClass.hpp,v 1.10 2007/02/08 21:06:44 paul Exp $
		 */
		class CClass
		{
		public:

//	===========================================================================

			const static long CCLASS_MAX_CLASSNAME_LENGTH = 128;				/**< Maximum length of class name */

//	===========================================================================

			/**
			 * Construction with the name of the class
			 * @param className The name of the class
			 */
			CClass(const char *className);

			/**
			 * Construction with the name of the class and its parent
			 * @param className The name of the class
			 * @param parentName The name of the parent
			 */
			CClass(const char *className, const char *parentName);

			/**
			 * Construction for use with templates when the parent is nothing
			 * @param templateName The name of the template (of the form Name<TypeName>)
			 * @param theType The type of the type (whatever it is)
			 */
			CClass(const char *templateName, const std::type_info &theType);

			/**
			 * Construction for use with templates when the parent is a counted object type
			 * @param templateName The name of the template (of the form Name<TypeName>)
			 * @param theType The type of the type (whatever it is)
			 * @param parentName The name of the parent
			 */
			CClass(const char *templateName, const std::type_info &theType, const char *parentName);

			/**
			 * Construction for use with templates when the parent is a template
			 * @param templateName The name of the template (of the form Name<TypeName>)
			 * @param theType The type of the type (whatever it is)
			 * @param parentName The name of the parent
			 * @param parentType The type of the type (whatever it is)
			 */
			CClass(const char *templateName, const std::type_info &theType, const char *parentName, const std::type_info &parentType);

			/**
			 * Default construction
			 */
			CClass();

			/**
			 * Copy construction
			 * @param theClass The class to copy
			 */
			CClass(const CClass &theClass);

			/**
			 * Destruction
			 */
			virtual ~CClass();

//	===========================================================================

			/**
			 * Comparison for a qsort
			 * @param class1 The first class
			 * @param class2 The second class
			 * @retval <0 class1 goes before class2 0 class1 == class2 >0 class1 goes after class2
			 * @see CClassManager::classCompareFunction
			 * @see TPointerCollection::qsortCompare
			 */
			static int compareClasses(const CClass **class1, const CClass **class2);

//	===========================================================================

			/**
			 * Assignment operator
			 * @param other The class to copy the name of
			 * @retval CClass& A reference to this
			 */
			CClass &operator = (const CClass &other);

			/**
			 * Equality operator
			 * @param other The class to check if this is equal to
			 * @retval bool if the classes have the same name, false otherwise
			 */
			bool operator == (const CClass &other) const;

			/**
			 * Increment the class count
			 * @note POSTFIX increment operator
			 * @note Class this function each time you construct your class
			 */
			void operator ++ (int /* Undefined */);

			/**
			 * Decrement the class count
			 * @note POSTFIX decrement operator
			 * @note Class this function each time you destruct your class
			 */
			void operator -- (int /* Undefined */);

//	===========================================================================

			/**
			 * Set the name of the class
			 * @param className The name of the class
			 */
			void setClassInformation(const char *className);

			/**
			 * Get the name of the class
			 * @retval const char* The class name
			 */
			const char *getClassName() const { return m_className; }

//	===========================================================================

			/**
			 * Check if this class is of a specific type.
			 * @note that this function is not a recommended function, due to the name mangling which may occur during template use\n
			 * although it is <B>usually</B> safe on CCountedObject inheritors that are not templates
			 * @see bool isTypeOf(const CClass &other) const
			 * @param className The name of the type of class you want to check that this is
			 * @retval bool True if class names are equal, false otherwise
			 */
			bool isTypeOf(const char *className) const;

			/**
			 * Check if this class is of a specific type
			 * @note This is purely a run time type information. Do not rely on this information to be cross platform available or even\n
			 * the same within different instanciations of the program. Inside the program you can rely on this information, however, dont assume that the name of the object\n
			 * in the code (for example CString) will be the same as the name in the class info. They may be different!\n
			 * Also note that this is the specific type of class ie if you have class a and class b : public a, this function will return false for:\n
			 * objectOfClassB->isTypeOf(b::getClass())
			 * @code
			 * if (myObject->getObjectClass().isTypeOf(SomeClass::getClass()))
			 * {
			 *     cout << "It is of the same type" << endl;
			 * }
			 * else
			 * {
			 *     cout << "It is of a different type" << endl;
			 * }
			 *
			 * // Or for templates
			 * if (myObject->getObjectClass().isTypeOf(SomeClass<CString>::getClass()))
			 * // Handle here
			 * @endcode
			 * @param other The class of the type that you want to check that this is
			 * @retval bool True if class names are equal, false otherwise
			 */
			bool isTypeOf(const CClass &other) const;

			/**
			 * Check if this class is a sub class of another class
			 * @note This is purely a run time type information. Do not rely on this information to be cross platform available or even\n
			 * the same within different instanciations of the program. Inside the program you can rely on this information, however, dont assume that the name of the object\n
			 * in the code (for example CString) will be the same as the name in the class info. They may be different!\n
			 * @code
			 * class a {  };
			 * class b : public a {  }
			 *
			 * b *myObject = new b;
			 *
			 * if (myObject->getObjectClass().isSubClassOf(a::getClass()))
			 * {
			 *     cout << "It is a sub type" << endl;
			 * }
			 * else
			 * {
			 *     cout << "It is not a sub type" << endl;
			 * }
			 * @endcode
			 * @param other The class of the type that you want to check that this is
			 * @retval bool True if class passed is a parent type of this
			 */
			bool isSubClassOf(const CClass &other) const;

			/**
			 * Get the current instance count
			 * @retval long The number of instances of this class that were created and not destroyed (ie still alive)
			 */
			long getCurrentInstanceCount() const { return m_instanceCount; }

			/**
			 * Get total number of this class constructed thus far
			 * @retval long The total number of classes created thus far
			 */
			long getTotalNumberOfClassConstructions() const { return m_maxInstanceCount; }

			/**
			 * Get the class manager
			 * @retval CClassManager* The class manager
			 */
			static CClassManager *getClassManager() { return CCLASSCLASS_MANAGER; }

			/**
			 * Delete the class manager
			 * @param filePath The path to write out the class information
			 */
			static void deleteClassManager(const char *filePath);

//	===========================================================================

		protected:

//	===========================================================================

			/**
			 * Compute a template typename combination
			 * @retval CClass The class that is constructed
			 * @param templateName The name of the template (of the form Name<TypeName>)
			 * @param theType The type of the type (whatever it is)
			 * @param theBuffer The buffer to write to
			 * @param bufferSize The size of the buffer
			 * @note NOT INTENDED FOR PUBLIC USE. THIS IS AN INTERNAL FUNCTION THAT IS REQUIRED TO BE PUBLIC BY ITS NATURE. YOU ARE NOT EXPECTED TO DIRECTLY CALL THIS\n
			 * UNLESS YOU ABSOLUTELY <B>TOTALLY</B> KNOW WHAT YOU ARE DOING
			 */
			static bool computeTemplateName(const char *templateName, const std::type_info &theType, char *theBuffer, const long bufferSize);

//	===========================================================================

			static CClassManager *CCLASSCLASS_MANAGER;			/**< Global class manager */

//	===========================================================================

			char m_className[CCLASS_MAX_CLASSNAME_LENGTH];		/**< Name of the class */
			char m_parentName[CCLASS_MAX_CLASSNAME_LENGTH];		/**< Name of the parent class */
			long m_instanceCount;								/**< Instances created and still alive */
			long m_maxInstanceCount;							/**< Total instances created */
		};
	}
}

/**
 * @def EXPONENT_CLASS_DECLARATION
 * Declares a class to be part of the Exponent class heirachy. Make sure that you place this define immediately after a classes opening brackets\n
 * It ensure that class descriptions are properly placed throughout the code.
 */
#define EXPONENT_CLASS_DECLARATION\
	public:\
		virtual const CClass &getObjectClass() const { return CCLASS_THE_CLASS; }\
		static const CClass &getClass() { return CCLASS_THE_CLASS; }\
	private:\
		static CClass CCLASS_THE_CLASS;

/**
 * @def EXPONENT_CLASS_IMPLEMENTATION_NO_PARENT(theClassName)
 * Place this at the top of your implementation file, passing the name of the class *without quotes* as the parameter
 */
#define EXPONENT_CLASS_IMPLEMENTATION_NO_PARENT(theClassName)\
	CClass theClassName::CCLASS_THE_CLASS(#theClassName);

/**
 * @def EXPONENT_CLASS_IMPLEMENTATION(theClassName, theParentName)
 * Place this at the top of your implementation file, passing the name of the class and the name of the parent class *without quotes* as the parameter
 * The parent name must be a valid class name for it to be used as a parent matchng string.
 */
#define EXPONENT_CLASS_IMPLEMENTATION(theClassName, theParentName)\
	CClass theClassName::CCLASS_THE_CLASS(#theClassName, #theParentName);

/**
 * @def EXPONENT_TEMPLATE_CLASS_IMPLEMENTATION_NO_PARENT(theClassName, TypeName)
 * Place this at the top of your implementation file, passing the name of the class *without quotes* as the parameter
 */
#define EXPONENT_TEMPLATE_CLASS_IMPLEMENTATION_NO_PARENT(theClassName, TypeName)\
	template<class TypeName> CClass theClassName::CCLASS_THE_CLASS(#theClassName, typeid(TypeName));

/**
 * @def EXPONENT_TEMPLATE_CLASS_IMPLEMENTATION(theClassName)
 * Place this at the top of your implementation file, passing the name of the class *without quotes* as the parameter
 */
#define EXPONENT_TEMPLATE_CLASS_IMPLEMENTATION(theClassName, TypeName, theParentName)\
	template<class TypeName> CClass theClassName::CCLASS_THE_CLASS(#theClassName, typeid(TypeName), #theParentName);

/**
 * @def EXPONENT_TEMPLATE_CLASS_IMPLEMENTATION_TEMPLATE_PARENT(theClassName, TypeName, theParentName, ParentTypeName)
 * Place this at the top of your implementation file, passing the name of the class *without quotes* as the parameter
 */
#define EXPONENT_TEMPLATE_CLASS_IMPLEMENTATION_TEMPLATE_PARENT(theClassName, TypeName, theParentName, ParentTypeName)\
	template<class TypeName> CClass theClassName::CCLASS_THE_CLASS(#theClassName, typeid(TypeName), #theParentName, typeid(ParentTypeName));

/**
 * @def EXPONENT_CLASS_CONSTRUCTION(theClassName)
 * Place this at the top of your constructor, passing the name of the class *without quotes* as the parameter
 */
#define EXPONENT_CLASS_CONSTRUCTION(theClassName)\
	theClassName::CCLASS_THE_CLASS++;

/**
 * @def EXPONENT_CLASS_DESTRUCTION(theClassName)
 * Place this at the top of your destructor, passing the name of the class *without quotes* as the parameter
 */
#define EXPONENT_CLASS_DESTRUCTION(theClassName)\
	theClassName::CCLASS_THE_CLASS--;

#endif // End of CClass.hpp